//
//	GSTabBarControl.h
//	GSTabBarControl
//
//	Created by John Pannell on 10/13/05.
//	Copyright 2005 Positive Spin Media. All rights reserved.
//

/*
 This view provides a control interface to manage a regular NSTabView.
 It looks and works like the tabbed browsing interface of many popular browsers.
 */

#import <Cocoa/Cocoa.h>
@class GSTabBarControl;

@protocol TabBarControlDelegate

//Standard NSTabView methods
- (BOOL)tabBar:(GSTabBarControl *)tabBarControl willUnselectTabItem:(NSViewController *)tabItem;
- (void)tabBar:(GSTabBarControl *)tabBarControl didSelectTabItem:(NSViewController *)tabItem;
- (BOOL)tabBar:(GSTabBarControl *)tabBarControl shouldCloseTabItem:(NSViewController *)tabItem;
- (void)tabBar:(GSTabBarControl *)tabBarControl didCloseTabItem:(NSViewController *)tabItem;
@optional
//"Spring-loaded" tabs methods
- (NSArray *)allowedDraggedTypesForContentView:(NSView *)aContentView;
- (void)tabBar:(GSTabBarControl *)tabBarControl acceptedDraggingInfo:(id<NSDraggingInfo>)draggingInfo onTabItem:(NSViewController *)tabItem;

//Contextual menu method
- (NSMenu *)tabView:(NSView *)aContentView menuForTabItem:(NSViewController *)tabItem;

//Drag and drop methods
- (BOOL)shouldDragTabItem:(NSViewController *)tabItem fromTabBar:(GSTabBarControl *)tabBarControl;
- (BOOL)shouldDropTabItem:(NSViewController *)tabItem inTabBar:(GSTabBarControl *)tabBarControl;
- (BOOL)shouldAllowTabItem:(NSViewController *)tabItem toLeaveTabBar:(GSTabBarControl *)tabBarControl;
- (void)didDropTabItem:(NSViewController *)tabItem inTabBar:(GSTabBarControl *)tabBarControl;

//Tear-off tabs methods
- (NSImage *)tabBar:(GSTabBarControl *)tabBarControl imageForTabItem:(NSViewController *)tabItem offset:(NSSize *)offset styleMask:(NSUInteger *)styleMask;
- (GSTabBarControl *)tabBar:(GSTabBarControl *)tabBarControl newTabBarForDraggedTabItem:(NSViewController *)tabItem atPoint:(NSPoint)point;
- (void)tabBar:(GSTabBarControl *)tabBarControl closeWindowForLastTabItem:(NSViewController *)tabItem;

//tab bar hiding methods
- (void)tabBarDidHide:(GSTabBarControl *)tabBarControl;
- (void)tabBarDidUnhide:(GSTabBarControl *)tabBarControl;
- (CGFloat)desiredWidthForVerticalTabBar:(GSTabBarControl *)tabBarControl;

// Adding
- (void)tabBar:(GSTabBarControl *)aTabBar didAddTabItem:(NSViewController *)tabItem;

//closing
- (void)tabBar:(GSTabBarControl *)aTabBar willCloseTabItem:(NSViewController *)tabItem;
- (BOOL)tabBar:(GSTabBarControl *)tabBarControl disableTabCloseForTabItem:(NSViewController *)tabItem;

//tooltips
- (NSString *)tabBar:(GSTabBarControl *)tabBarControl toolTipForTabItem:(NSViewController *)tabItem;

//accessibility
- (NSString *)accessibilityStringForTabBar:(GSTabBarControl *)tabBarControl objectCount:(NSInteger)objectCount;

@end

#define PSMTabDragDidEndNotification @"PSMTabDragDidEndNotification"
#define PSMTabDragDidBeginNotification @"PSMTabDragDidBeginNotification"

// internal cell border
#define MARGIN_X 24
#define MARGIN_Y 5
// padding between objects
#define kPSMTabBarCellPadding 16
// fixed size objects
#define kPSMMinimumTitleWidth 30

#define kPSMHideAnimationSteps 3.0

// Value used in _currentStep to indicate that resizing operation is not in progress
#define kPSMIsNotBeingResized -1

// Value used in _currentStep when a resizing operation has just been started
#define kPSMStartResizeAnimation 0

@protocol PSMTabStyle;

typedef enum {
	PSMTabBarHorizontalOrientation,
	PSMTabBarVerticalOrientation
} PSMTabBarOrientation;

typedef enum {
	PSMTabBarTearOffAlphaWindow,
	PSMTabBarTearOffMiniwindow
} PSMTabBarTearOffStyle;

enum {
	PSMTab_SelectedMask = 1 << 1,
	PSMTab_LeftIsSelectedMask = 1 << 2,
	PSMTab_RightIsSelectedMask = 1 << 3,
	PSMTab_PositionLeftMask = 1 << 4,
	PSMTab_PositionMiddleMask = 1 << 5,
	PSMTab_PositionRightMask = 1 << 6,
	PSMTab_PositionSingleMask = 1 << 7
};

@interface GSTabBarControl : NSControl <NSAnimationDelegate, NSMenuDelegate> {}
@property (nonatomic, weak) IBOutlet NSViewController *controller;
@property (nonatomic, weak) IBOutlet id delegate;
@property (nonatomic, unsafe_unretained) IBOutlet NSView *contentView;
@property (nonatomic, strong, getter=tabItems) NSMutableArray *viewControllers;
@property (nonatomic, readonly) NSUInteger selectionIndex;
@property (nonatomic, readonly) NSUInteger countOfTabItems;

@property (nonatomic) CGFloat scrollOffset;

@property (nonatomic, unsafe_unretained, setter=selectTabItem:) NSViewController *selectedTabItem;
// settings
@property (nonatomic) PSMTabBarOrientation orientation;
@property (nonatomic) BOOL canCloseOnlyTab;
@property (nonatomic) BOOL allowsBackgroundTabClosing;
@property (nonatomic) BOOL allowsScrubbing;
@property (nonatomic) BOOL sizeCellsToFit;
@property (nonatomic) BOOL useFullWidth;
@property (nonatomic) BOOL hideForSingleTab;
@property (nonatomic) BOOL drawTopDivider;
@property (nonatomic) NSInteger cellMaxWidth;
@property (nonatomic, strong) NSString *styleName;

@property (nonatomic, readonly) NSAppearance *contentAppearance;

@property (nonatomic) BOOL disableUpdates;

// always show first tab when scrolling
@property (nonatomic) BOOL pinFirstTab;

- (void)addTabItem:(NSViewController *)item;
- (NSViewController *)tabItemAtIndex:(NSInteger)idx;

- (void)insertTabItem:(NSViewController *)item atIndex:(NSUInteger)idx;
- (void)removeTabItem:(NSViewController *)item;
- (void)closeTabItem:(NSViewController *)item;

- (void)selectTabItemAtIndex:(NSUInteger)idx;
- (NSUInteger)indexOfTabItem:(NSViewController *)tab;
- (NSInteger)numberOfVisibleTabs;

- (NSUInteger)indexOfCell:(id)cell;

- (void)tabClick:(id)sender;

- (void)update;

- (NSRect)genericCellRect;
@end
